#include <string>

#include "../../include/ring.h"
#include "../../include/RationalNumberPolynomial/urpolynomial.h"
#include "../../include/RingPolynomial/upolynomial.h"
#include "RationalFunction/rationalfunction_euclideanmethods.h"

template <class UnivariatePolynomialOverField, class Field>
void _euclideanDivide(UnivariatePolynomialOverField &a, UnivariatePolynomialOverField &b, UnivariatePolynomialOverField *q, UnivariatePolynomialOverField *r){
	if (b.isZero())
		throw std::invalid_argument( "euclideanDivide: attempt to divide by zero polynomial" );

	Integer delta;
	Field c;
	UnivariatePolynomialOverField t;
	UnivariatePolynomialOverField temp;
	t.setVariableName(a.variable());
	temp.setVariableName(a.variable());

	q->zero();
	*r = a;
	delta = r->degree()-b.degree();
	while (!(r->isZero()) && delta >= 0){
		c = r->leadingCoefficient();
		c /= b.leadingCoefficient();
		t.zero();
		t = (t + c << delta.get_si()); // t = c*x^delta
		*q += t;
		temp = b;
		temp *= t;
		*r -= temp;
		delta = r->degree()-b.degree();
	}
}

template <class UnivariatePolynomialOverField, class Field>
void _quotient(UnivariatePolynomialOverField &a, UnivariatePolynomialOverField &b, UnivariatePolynomialOverField *q){
	if (b.isZero())
		throw std::invalid_argument( "quotient: attempt to divide by zero polynomial" );

	Integer delta;
	Field c;
	UnivariatePolynomialOverField t,temp;
	UnivariatePolynomialOverField r(a);
	t.setVariableName(a.variable());
	temp.setVariableName(a.variable());

	delta = r.degree()-b.degree();
	while (!(r.isZero()) && delta >= 0){
		c = r.leadingCoefficient();
		c /= b.leadingCoefficient();
		t.zero();
		t = (t + c << delta.get_si()); // t = c*x^delta
		*q += t;
		temp = b;
		temp *= t;
		r -= temp;
		delta = r.degree()-b.degree();
	}
}

template <class UnivariatePolynomialOverField, class Field>
void _remainder(UnivariatePolynomialOverField &a, UnivariatePolynomialOverField &b, UnivariatePolynomialOverField *r){
	if (b.isZero())
		throw std::invalid_argument( "remainder: attempt to divide by zero polynomial" );

	Integer delta;
	Field c;
	UnivariatePolynomialOverField t,temp,q;
	t.setVariableName(a.variable());
	temp.setVariableName(a.variable());
	q.setVariableName(a.variable());

	*r = a;
	delta = r->degree()-b.degree();
	while (!(r->isZero()) && delta >= 0){
		c = r->leadingCoefficient();
		c /= b.leadingCoefficient();
		t.zero();
		t = (t + c << delta.get_si()); // t = c*x^delta
		q += t;
		temp = b;
		temp *= t;
		*r -= temp;
		delta = r->degree()-b.degree();
	}
}

template <class UnivariatePolynomialOverField, class Field>
void _halfExtendedEuclidean(UnivariatePolynomialOverField &A, UnivariatePolynomialOverField &B, UnivariatePolynomialOverField *s, UnivariatePolynomialOverField *g){
	/* halfExtendendEuclidean(a,b,s,g)                   */
	/* Given UPoF A and B, return UPoF s and g such that */
	/* g = gcd(A,B) and s*a equiv g (mod B)              */

	UnivariatePolynomialOverField a(A);	// a = A
	UnivariatePolynomialOverField b(B);	// b = B
	UnivariatePolynomialOverField q,r,r1,a1,b1;
	b.setVariableName(A.variable());
	q.setVariableName(A.variable());
	r.setVariableName(A.variable());
	r1.setVariableName(A.variable());
	a1.setVariableName(A.variable());
	b1.setVariableName(A.variable());
	Field c;

	a1.one();
	b1.zero();
	while (!b.isZero()){
		_euclideanDivide<UnivariatePolynomialOverField,Field>(a,b,&q,&r);	// a = b*q + r
		a = b;
		b = r;
		r1 = -b1;
		r1 *= q;
		r1 += a1;
		a1 = b1;
		b1 = r1;
	}
	c = a.leadingCoefficient();
	a1 /= c;
	a /= c;
	*s = a1;
	*g = a;
}

template <class UnivariatePolynomialOverField, class Field>
void _halfExtendedEuclidean(UnivariatePolynomialOverField &a, UnivariatePolynomialOverField &b, UnivariatePolynomialOverField &c, UnivariatePolynomialOverField *s){
	/* halfExtendendEuclidean(a,b,c,s) - diophantine version */
	/* Given UPoF a, b and c, with c in the ideal (a,b),     */
	/* return UPof s such that s*a equiv c (mod b) and       */
	/* either s = 0 or degree(s) < degree(b)                 */

	UnivariatePolynomialOverField g,q,r;
	g.setVariableName(a.variable());
	q.setVariableName(a.variable());
	r.setVariableName(a.variable());

	_halfExtendedEuclidean<UnivariatePolynomialOverField,Field>(a,b,s,&g);	// s*a equiv g (mod b)
	_euclideanDivide<UnivariatePolynomialOverField,Field>(c,g,&q,&r);			// c = gq + r
	if (!(r.isZero()))
		throw std::invalid_argument( "third arg is not in the ideal generated by first two args" );
	*s *= q;
	if (!(s->isZero()) && s->degree() >= b.degree()){
		_euclideanDivide<UnivariatePolynomialOverField,Field>(*s,b,&q,&r);	// s = b*q + r
		*s = r;
	}
}

template <class UnivariatePolynomialOverField, class Field>
void _extendedEuclidean(UnivariatePolynomialOverField &A, UnivariatePolynomialOverField &B, UnivariatePolynomialOverField *s, UnivariatePolynomialOverField *t, UnivariatePolynomialOverField *g){
	/* extendendEuclidean(a,b,c,s,t)                       */
	/* Given UPoF a, b, return UPoF s, t, g, such that     */
	/* g = gcd(A,B) and s*A + t*B = g.                     */

	UnivariatePolynomialOverField temp,temp2;
	temp.setVariableName(A.variable());
	temp2.setVariableName(A.variable());

	_halfExtendedEuclidean<UnivariatePolynomialOverField,Field>(A,B,s,g);		// s*A equiv g (mod B)
	temp = *s;
	temp *= -A;
	temp += *g;							// temp = g - s*A
	_euclideanDivide<UnivariatePolynomialOverField,Field>(temp,B,t,&temp2);	// temp2 must be 0
}

template <class UnivariatePolynomialOverField, class Field>
void _extendedEuclidean(UnivariatePolynomialOverField &A, UnivariatePolynomialOverField &B, UnivariatePolynomialOverField &C, UnivariatePolynomialOverField *s, UnivariatePolynomialOverField *t){
	/* extendendEuclidean(a,b,c,s,t) - diophantine version */
	/* Given DenseUnivariateRationalPolynomial a, b and c, with c in the ideal (a,b),   */
	/* return DenseUnivariateRationalPolynomial s and t such that s*a + t*b = c and     */
	/* either s = 0 or degree(s) < degree(b)               */

	UnivariatePolynomialOverField temp;
	temp.setVariableName(A.variable());

	_halfExtendedEuclidean<UnivariatePolynomialOverField,Field>(A,B,C,s);		// s*A equiv C (mod B)
	temp = *s;
	temp *= -A;
	temp += C;
	_euclideanDivide<UnivariatePolynomialOverField,Field>(temp,B,t,&temp);	// returned temp must be 0
}

// to avoid linking errors
template void _euclideanDivide<DenseUnivariateRationalPolynomial,RationalNumber>(DenseUnivariateRationalPolynomial &a, DenseUnivariateRationalPolynomial &b, DenseUnivariateRationalPolynomial *q, DenseUnivariateRationalPolynomial *r);
template void _euclideanDivide<SparseUnivariatePolynomial<RationalNumber>,RationalNumber>(SparseUnivariatePolynomial<RationalNumber> &a, SparseUnivariatePolynomial<RationalNumber> &b, SparseUnivariatePolynomial<RationalNumber> *q, SparseUnivariatePolynomial<RationalNumber> *r);
template void _euclideanDivide<SparseUnivariatePolynomial<ComplexRationalNumber>,ComplexRationalNumber>(SparseUnivariatePolynomial<ComplexRationalNumber> &a, SparseUnivariatePolynomial<ComplexRationalNumber> &b, SparseUnivariatePolynomial<ComplexRationalNumber> *q, SparseUnivariatePolynomial<ComplexRationalNumber> *r);

template void _quotient<DenseUnivariateRationalPolynomial,RationalNumber>(DenseUnivariateRationalPolynomial &a, DenseUnivariateRationalPolynomial &b, DenseUnivariateRationalPolynomial *q);
template void _quotient<SparseUnivariatePolynomial<RationalNumber>,RationalNumber>(SparseUnivariatePolynomial<RationalNumber> &a, SparseUnivariatePolynomial<RationalNumber> &b, SparseUnivariatePolynomial<RationalNumber> *q);
template void _quotient<SparseUnivariatePolynomial<ComplexRationalNumber>,ComplexRationalNumber>(SparseUnivariatePolynomial<ComplexRationalNumber> &a, SparseUnivariatePolynomial<ComplexRationalNumber> &b, SparseUnivariatePolynomial<ComplexRationalNumber> *q);

template void _remainder<DenseUnivariateRationalPolynomial,RationalNumber>(DenseUnivariateRationalPolynomial &a, DenseUnivariateRationalPolynomial &b, DenseUnivariateRationalPolynomial *r);
template void _remainder<SparseUnivariatePolynomial<RationalNumber>,RationalNumber>(SparseUnivariatePolynomial<RationalNumber> &a, SparseUnivariatePolynomial<RationalNumber> &b, SparseUnivariatePolynomial<RationalNumber> *r);
template void _remainder<SparseUnivariatePolynomial<ComplexRationalNumber>,ComplexRationalNumber>(SparseUnivariatePolynomial<ComplexRationalNumber> &a, SparseUnivariatePolynomial<ComplexRationalNumber> &b, SparseUnivariatePolynomial<ComplexRationalNumber> *r);

template void _halfExtendedEuclidean<DenseUnivariateRationalPolynomial,RationalNumber>(DenseUnivariateRationalPolynomial &A, DenseUnivariateRationalPolynomial &B, DenseUnivariateRationalPolynomial *s, DenseUnivariateRationalPolynomial *g);
template void _halfExtendedEuclidean<SparseUnivariatePolynomial<RationalNumber>,RationalNumber>(SparseUnivariatePolynomial<RationalNumber> &A, SparseUnivariatePolynomial<RationalNumber> &B, SparseUnivariatePolynomial<RationalNumber> *s, SparseUnivariatePolynomial<RationalNumber> *g);
template void _halfExtendedEuclidean<SparseUnivariatePolynomial<ComplexRationalNumber>,ComplexRationalNumber>(SparseUnivariatePolynomial<ComplexRationalNumber> &A, SparseUnivariatePolynomial<ComplexRationalNumber> &B, SparseUnivariatePolynomial<ComplexRationalNumber> *s, SparseUnivariatePolynomial<ComplexRationalNumber> *g);

template void _halfExtendedEuclidean<DenseUnivariateRationalPolynomial,RationalNumber>(DenseUnivariateRationalPolynomial &a, DenseUnivariateRationalPolynomial &b, DenseUnivariateRationalPolynomial &c, DenseUnivariateRationalPolynomial *s);
template void _halfExtendedEuclidean<SparseUnivariatePolynomial<RationalNumber>,RationalNumber>(SparseUnivariatePolynomial<RationalNumber> &a, SparseUnivariatePolynomial<RationalNumber> &b, SparseUnivariatePolynomial<RationalNumber> &c, SparseUnivariatePolynomial<RationalNumber> *s);
template void _halfExtendedEuclidean<SparseUnivariatePolynomial<ComplexRationalNumber>,ComplexRationalNumber>(SparseUnivariatePolynomial<ComplexRationalNumber> &a, SparseUnivariatePolynomial<ComplexRationalNumber> &b, SparseUnivariatePolynomial<ComplexRationalNumber> &c, SparseUnivariatePolynomial<ComplexRationalNumber> *s);

template void _extendedEuclidean<DenseUnivariateRationalPolynomial,RationalNumber>(DenseUnivariateRationalPolynomial &A, DenseUnivariateRationalPolynomial &B, DenseUnivariateRationalPolynomial *s, DenseUnivariateRationalPolynomial *t, DenseUnivariateRationalPolynomial *g);
template void _extendedEuclidean<SparseUnivariatePolynomial<RationalNumber>,RationalNumber>(SparseUnivariatePolynomial<RationalNumber> &A, SparseUnivariatePolynomial<RationalNumber> &B, SparseUnivariatePolynomial<RationalNumber> *s, SparseUnivariatePolynomial<RationalNumber> *t, SparseUnivariatePolynomial<RationalNumber> *g);
template void _extendedEuclidean<SparseUnivariatePolynomial<ComplexRationalNumber>,ComplexRationalNumber>(SparseUnivariatePolynomial<ComplexRationalNumber> &A, SparseUnivariatePolynomial<ComplexRationalNumber> &B, SparseUnivariatePolynomial<ComplexRationalNumber> *s, SparseUnivariatePolynomial<ComplexRationalNumber> *t, SparseUnivariatePolynomial<ComplexRationalNumber> *g);

template void _extendedEuclidean<DenseUnivariateRationalPolynomial,RationalNumber>(DenseUnivariateRationalPolynomial &A, DenseUnivariateRationalPolynomial &B, DenseUnivariateRationalPolynomial &C, DenseUnivariateRationalPolynomial *s, DenseUnivariateRationalPolynomial *t);
template void _extendedEuclidean<SparseUnivariatePolynomial<RationalNumber>,RationalNumber>(SparseUnivariatePolynomial<RationalNumber> &A, SparseUnivariatePolynomial<RationalNumber> &B, SparseUnivariatePolynomial<RationalNumber> &C, SparseUnivariatePolynomial<RationalNumber> *s, SparseUnivariatePolynomial<RationalNumber> *t);
template void _extendedEuclidean<SparseUnivariatePolynomial<ComplexRationalNumber>,ComplexRationalNumber>(SparseUnivariatePolynomial<ComplexRationalNumber> &A, SparseUnivariatePolynomial<ComplexRationalNumber> &B, SparseUnivariatePolynomial<ComplexRationalNumber> &C, SparseUnivariatePolynomial<ComplexRationalNumber> *s, SparseUnivariatePolynomial<ComplexRationalNumber> *t);
