
#ifndef _ASYNC_GENERATOR_THREAD_HPP_
#define _ASYNC_GENERATOR_THREAD_HPP_

#if defined(SERIAL) && SERIAL
#define ASYNC_GEN_SERIAL 1
#define ASYNC_GEN_PARALLEL 0
#else 
#define ASYNC_GEN_SERIAL 0
#define ASYNC_GEN_PARALLEL 1
#endif

#if ASYNC_GEN_PARALLEL
#include "AsyncObjectStream.hpp"
#endif

#include "AsyncGenerator.hpp"

#include <functional>
#include <queue>

#include <sstream>


/**
 * A class to encapsulate the asynchronous generation of objects from some
 * function. The general scheme is that some "producer" function will 
 * add results to this generator via the generateObject method.
 * Then, the consumer will retrieve objects that were generated via the getNextObject
 * method. 
 *
 * This thread always spawns a new thread (unless SERIAL is 1) on construction.
 * This constrasts with the other implementation in AsyncGenerator.hpp
 * which goes serial if the pool runs out.
 *
 * Note1: All functions passed to the generator as "producer" functions should take, 
 * as their final parameter, an AsyncGenerator reference. This allows the producer
 * to obtain a shared instance of the AsyncGenerator and use it to generate
 * the objects. This AsyncGenerator should *not* appear in the last of arguments
 * passed to the AsyncGenerator constructor; it will automatically be inserted.
 * 
 * Note2: this class need not necessarily be asynchronous, say if wishing to build
 * in a serial fashion. In that case the interface remains the same but is the 
 * generation of objects is not done serially. 
 *
 * 
 */
template <class Object>
class AsyncGeneratorThread : public AsyncGenerator<Object> {

#if ASYNC_GEN_PARALLEL
	AsyncObjectStream<Object> stream;
	std::thread m_thread;
	std::recursive_mutex m_mutex;
#else
	//queue used either by a SERIAL build or when max threads reached.
	std::queue<Object> results;
	bool finishedProducing;
#endif

	
	/**
	 * Join the generation method to the current thread. 
	 * Effectively waits for all possible objects to be generated.
	 */
	void join() {
#if ASYNC_GEN_PARALLEL
		if (m_thread.joinable()) {
			m_thread.join();
		}
#else 
#endif
	}

	/**
	 * Detch the generation method from the current thread.
	 * Allows the generation to occur in a detached way from the main thead.
	 */
	void detach() {
#if ASYNC_GEN_PARALLEL
		if (m_thread.joinable()) {
			m_thread.detach();
		}
#else 
#endif
	}


public:


	//Tricky tricky here. Passing this, which is not fully constructed yet, 
	//to the function. If this class ever gets more complicated, can cause
	//weird errors in order of initialization.
	/**
	 * Construct an AsyncGenerator given some "producer" function and the arguments
	 * to pass to that function.
	 *
	 * All functions passed should take, as their final parameter, 
	 * an AsyncGenerator reference. This allows the producer
	 * to obtain a shared instance of the AsyncGenerator and use it to generate
	 * the objects. An AsyncGenerator reference should not be passed in the list of args.
	 * An AsyncGenerator reference will automatically be inserted.
	 *
	 */
	template <class Function, class... Args>
	explicit AsyncGeneratorThread(Function&& f, Args&&...args) :
#if ASYNC_GEN_PARALLEL
		stream(),
		m_thread()
#else
		finishedProducing(0)	
#endif
	{
#if ASYNC_GEN_PARALLEL
		m_thread = std::thread(std::forward<Function>(f), std::forward<Args>(args)..., std::ref(*this));
#else
		auto boundF = std::bind(f, args..., std::ref(*this));
		boundF();
		finishedProducing = 1;
#endif
	}


	/**
	 * Create an empty AsyncGenerator that will never produce anything
	 * since it has not producing method.
	 */
	AsyncGeneratorThread() :
#if ASYNC_GEN_PARALLEL
		stream(),
		m_thread()
#else
		finishedProducing(1)
#endif
	{

	}

	~AsyncGeneratorThread() {
#if ASYNC_GEN_PARALLEL
		this->setComplete();
		this->detach();
#endif
	}

	/**
	 * Add a new object to the generated by this AsyncGenerator.
	 * Note that the passed Object is moved from.
	 *
	 * obj: the Object to generate.
	 */
	void generateObject(Object& obj) {
#if ASYNC_GEN_PARALLEL
		stream.addResult(obj);
#else
		results.push(std::move(obj));
#endif
	}

	/**
	 * Add a new object to the generated by this AsyncGenerator.
	 * Note that the passed Object is moved from.
	 *
	 * obj: the Object to generate.
	 */
	void generateObject(const Object& obj) {
#if ASYNC_GEN_PARALLEL
		Object objTemp(obj);
		stream.addResult(objTemp);
#else
		results.push(obj);
#endif
	}

	/**
	 * Finalize the AsyncGenerator by the producer declaring it has finished
	 * generating all possible objects.
	 */
	void setComplete() {
#if ASYNC_GEN_PARALLEL
		stream.resultsFinished();
#else 
		finishedProducing = 1;
#endif
	}

	/**
	 * Obtain an Object which was generated.
	 * The generated object is returned in obj.
	 * 
	 * returns false iff no more objects could be obtained and
	 * no more objects will ever be generated.
	 */
	bool getNextObject(Object& obj) {
#if ASYNC_GEN_PARALLEL
		return stream.getNextObject(obj);
#else 
		if(finishedProducing && results.empty()) {
			return false;
		}
		obj = std::move(results.front());
		results.pop();
		return true;
#endif

	}

	bool isFinished() {
#if ASYNC_GEN_PARALLEL
		return stream.isFinished();
#else
		if(finishedProducing && results.empty()) {
			return true;
		} else {
			return false;
		}
#endif
	}


};




#endif
