
#ifndef _ASYNC_GENERATOR_POOL_HPP_
#define _ASYNC_GENERATOR_POOL_HPP_

#if defined(SERIAL) && SERIAL
#define ASYNC_GEN_SERIAL 1
#define ASYNC_GEN_PARALLEL 0
#else
#define ASYNC_GEN_SERIAL 0
#define ASYNC_GEN_PARALLEL 1
#endif

#if ASYNC_GEN_PARALLEL
#include "AsyncObjectStream.hpp"
#include "ExecutorThreadPool.hpp"
#endif

#include "AsyncGenerator.hpp"

#include <functional>
#include <queue>



/**
 * A class to encapsulate the asynchronous generation of objects from some
 * function. The general scheme is that some "producer" function will
 * add results to this generator via the generateObject method.
 * Then, the consumer will retrieve objects that were generated via the getNextObject
 * method.
 *
 * Note1: All functions passed to the generator as "producer" functions should take,
 * as their final parameter, an AsyncGenerator reference. This allows the producer
 * to obtain a shared instance of the AsyncGenerator and use it to generate
 * the objects. This AsyncGenerator should *not* appear in the last of arguments
 * passed to the AsyncGenerator constructor; it will automatically be inserted.
 *
 * Note2: this class need not necessarily be asynchronous, say if wishing to build
 * in a serial fashion. In that case the interface remains the same but is the
 * generation of objects is not done serially.
 *
 *
 */
template <class Object>
class AsyncGeneratorPool : public AsyncGenerator<Object> {

#if ASYNC_GEN_PARALLEL
	AsyncObjectStream<Object> stream;
	bool executingAsync;
	std::recursive_mutex m_mutex;
#endif

	//queue used either by a SERIAL build or when max threads reached.
	std::queue<Object> results;
	bool finishedProducing;

public:


	//Tricky tricky here. Passing this, which is not fully constructed yet,
	//to the function. If this class ever gets more complicated, can cause
	//weird errors in order of initialization.
	/**
	 * Construct an AsyncGenerator given some "producer" function and the arguments
	 * to pass to that function.
	 *
	 * All functions passed should take, as their final parameter,
	 * an AsyncGenerator reference. This allows the producer
	 * to obtain a shared instance of the AsyncGenerator and use it to generate
	 * the objects. An AsyncGenerator reference should not be passed in the list of args.
	 * An AsyncGenerator reference will automatically be inserted.
	 *
 	 * @param f: the producer function
	 * @param args: the arguments to the producer function
	 */
	template <class Function, class... Args>
	explicit AsyncGeneratorPool(Function&& f, Args&&...args) :
#if ASYNC_GEN_PARALLEL
		stream(),
		executingAsync(0),
#endif
		finishedProducing(0)
	{
#if ASYNC_GEN_PARALLEL
		std::function<void()> boundF = std::bind(std::forward<Function>(f), std::forward<Args>(args)..., std::ref(*this));
		if (ExecutorThreadPool::getThreadPool().allThreadsBusy()) {
			boundF();
		} else {
			executingAsync = 1;
			ExecutorThreadPool::getThreadPool().addTask(boundF);
		}
#endif
#if ASYNC_GEN_SERIAL
		auto boundF = std::bind(f, args..., std::ref(*this));
		boundF();
		finishedProducing = 1;
#endif
	}



#if ASYNC_GEN_PARALLEL
	/**
	 * Construct an AsyncGenerator to execute on a pre-determined
	 * FunctionExecutorThread given by the threadID tid.
	 * The producer function to execute is passed as f, with arguments args.
	 *
	 * All functions passed should take, as their final parameter,
	 * an AsyncGenerator reference. This allows the producer
	 * to obtain a shared instance of the AsyncGenerator and use it to generate
	 * the objects. An AsyncGenerator reference should not be passed in the list of args.
	 * An AsyncGenerator reference will automatically be inserted.
	 *
	 * @param tid: the thread id to execute the producer function
	 * @param f: the producer function
	 * @param args: the arguments to the producer function
	 *
	 */
	template <class Function, class... Args>
	explicit AsyncGeneratorPool(ExecutorThreadPool::threadID tid, Function&& f, Args&&...args) :
		stream(),
		executingAsync(1),
		finishedProducing(0)
	{
		std::function<void()> boundF = std::bind(std::forward<Function>(f), std::forward<Args>(args)..., std::ref(*this));
		ExecutorThreadPool::getThreadPool().executeTask(tid, boundF);
	}
#endif

	AsyncGeneratorPool(std::vector<Object> objects) :
		stream(),
		executingAsync(0),
		results(),
		finishedProducing(1)
	{
		for (Object& o : objects) {
			results.push(std::move(o));
		}
	}


	/**
	 * Create an empty AsyncGenerator that will never produce anything
	 * since it has not producing method.
	 */
	AsyncGeneratorPool() :
#if ASYNC_GEN_PARALLEL
		stream(),
		executingAsync(0),
#endif
		finishedProducing(1)
	{

	}

	~AsyncGeneratorPool() {
#if ASYNC_GEN_PARALLEL
		this->setComplete();
#endif
	}

	/**
	 * Add a new object to the generated by this AsyncGenerator.
	 * Note that the passed Object is moved from.
	 *
	 * obj: the Object to generate.
	 */
	void generateObject(Object& obj) {
#if ASYNC_GEN_PARALLEL
		if (executingAsync) {
			stream.addResult(obj);
		} else {
#endif
			results.push(std::move(obj));

#if ASYNC_GEN_PARALLEL
		}
#endif
	}

	/**
	 * Add a new object to the generated by this AsyncGenerator.
	 *
	 * obj: the Object to generate.
	 */
	void generateObject(const Object& obj) {
#if ASYNC_GEN_PARALLEL
		if (executingAsync) {
			Object objTemp(obj);
			stream.addResult(objTemp);
		} else {
#endif
			results.push(obj);

#if ASYNC_GEN_PARALLEL
		}
#endif
	}

	/**
	 * Finalize the AsyncGenerator by the producer declaring it has finished
	 * generating all possible objects.
	 */
	void setComplete() {
#if ASYNC_GEN_PARALLEL
		if (executingAsync) {
			stream.resultsFinished();
		} else {
#endif
			finishedProducing = 1;

#if ASYNC_GEN_PARALLEL
		}
#endif
	}

	/**
	 * Obtain an Object which was generated.
	 * The generated object is returned in obj.
	 *
	 * returns false iff no more objects could be obtained and
	 * no more objects will ever be generated.
	 */
	bool getNextObject(Object& obj) {
#if ASYNC_GEN_PARALLEL
		if (executingAsync) {
			return stream.getNextObject(obj);
		} else {
#endif
			if(finishedProducing && results.empty()) {
				return false;
			}
			obj = std::move(results.front());
			results.pop();
			return true;
#if ASYNC_GEN_PARALLEL
		}
#endif

	}

	bool isFinished() {
#if ASYNC_GEN_PARALLEL
		if (executingAsync) {
			return stream.isFinished();
		} else {
#endif
			if(finishedProducing && results.empty()) {
				return true;
			} else {
				return false;
			}
#if ASYNC_GEN_PARALLEL
		}
#endif
	}

};




#endif
