
#ifndef _ASYNC_GENERATOR_HPP_
#define _ASYNC_GENERATOR_HPP_

/**
 * A class to encapsulate the asynchronous generation of objects from some
 * function. The general scheme is that some "producer" function will 
 * add results to this generator via the generateObject method.
 * Then, the consumer will retrieve objects that were generated via the getNextObject
 * method. 
 */
template <class Object>
class AsyncGenerator {

public:

	/**
	 * Add a new object to the generated by this AsyncGenerator.
	 * Note that the passed Object is moved from.
	 *
	 * obj: the Object to generate.
	 */
	virtual void generateObject(Object& obj) = 0;

	/**
	 * Add a new object to the generated by this AsyncGenerator.
	 * Note that the passed Object is moved from.
	 *
	 * obj: the Object to generate.
	 */
	virtual void generateObject(const Object& obj) = 0;

	/**
	 * Finalize the AsyncGenerator by the producer declaring it has finished
	 * generating all possible objects.
	 */
	virtual void setComplete() = 0;

	/**
	 * Obtain an Object which was generated.
	 * The generated object is returned in obj.
	 * 
	 * returns false iff no more objects could be obtained and
	 * no more objects will ever be generated.
	 */
	virtual bool getNextObject(Object& obj) = 0;

	/**
	 * Determine if the AsyncGenerator has finished producing
	 * and if all produced objects have been retrieved.
	 * 
	 * returns true iff both the generator is done produced and all objects have been retrieved.
	 */
	virtual bool isFinished() = 0;

};




#endif
